/* $Id: main.cpp,v 1.3 2007-11-08 15:43:40 aehyvari Exp $ */
/* start[./copyright.txt] */
/* Including data from file ./copyright.txt */
/* file opened */
/* Reading 775 bytes of data */
/* Start of include */
/*
 * Copyright 2007 Antti Hyvrinen
 *
 * This file is part of GridJM.
 *
 * GridJM is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * GridJM is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GridJM; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/* End of include */
/* stop[./copyright.txt] */
/* Main of gridjm.
 * Parse args, and start the gridjm side and the userside. Main thread
 * becomes the gridjm, and userside is a new thread
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "gridjm.h"
#include "parseargs.h"
#include "userside.h"

//using namespace std;

void child_signal(int arg) {
    signal (SIGCHLD, child_signal);
}

void pipe_signal(int arg) {
    signal (SIGPIPE, pipe_signal);
    printf("Received SIGPIPE\n");
}

int check(struct arguments * args) {
    int rval = 0;
    char* dir = args->dir;
    // Check the download directory
    struct stat *buf = (struct stat *)malloc(sizeof(struct stat));
    stat(args->dir, buf);
    mode_t mode  = buf->st_mode;

    // Not a directory
    if (! S_ISDIR(mode)) {
        printf("%s is not a directory\n", dir);
        rval = 1;
    }

    else if (access(dir, W_OK) == -1) {
        printf("Error accessing %s:\n", dir);
        perror("access()");
        rval = 1;
    }

    free(buf);

    if (rval != 0)
        printf("Refusing to start\n");
    return rval;
}

int main(int argc, char **argv) {

    struct arguments * args =
        (struct arguments *) malloc(sizeof(struct arguments));

    // Ignore SIGPIPE, since the network routines will generate it
    signal(SIGPIPE, pipe_signal);

    parseargs(argc, argv, args);

    if (args->error) {
        cerr << "Exiting" << endl;
        free(args);
        return 1;
    }

    // Output pid to the pid file
    pid_t pid = getpid();
    char *pidstr = (char *)malloc(7);
    sprintf(pidstr, "%i\n", pid);
    int fd = open(args->pidfile, O_WRONLY | O_CREAT, S_IRUSR | S_IWUSR);
    if (fd < 0) {
        cerr << "Error opening pid file " << args->pidfile << endl;
        return 1;
    }
    int i = write(fd, pidstr, strlen(pidstr));
    close(fd);
    free(pidstr);

    // Block the child exit signal
    sigset_t *sigmask = (sigset_t *)malloc(sizeof(sigset_t));
    sigset_t *orig_sigmask = (sigset_t *)malloc(sizeof(sigset_t));

    sigemptyset(sigmask);
    sigaddset(sigmask, SIGCHLD);
    sigprocmask(SIG_BLOCK, sigmask, orig_sigmask);
    // Attach a handler so that we can see them when they are unblocked
    signal(SIGCHLD, child_signal);

    Instrumenter* inst = new Instrumenter(args->instrumentation);

    queue<char*> *new_jobs = new queue<char*>();
    pthread_mutex_t *new_jobs_lock =
        (pthread_mutex_t *) malloc(sizeof(pthread_mutex_t));
    pthread_mutex_init(new_jobs_lock, 0);

    list<PidJob*> *forked_downloads = new list<PidJob*>();
    pthread_mutex_t *forked_dl_lock =
        (pthread_mutex_t *) malloc(sizeof(pthread_mutex_t));
    pthread_mutex_init(forked_dl_lock, 0);

    int pipe_write = -1;
    int pipe_read = 1;
    int filedes[2];

    if ( pipe(filedes) )
        perror("pipe()");

    // Make a sanity check of arguments
    if (check(args) != 0)
        return 1;

    GridJM jm = GridJM(inst, args, filedes[1], new_jobs, new_jobs_lock,
            forked_downloads, forked_dl_lock);

    UserSide us = UserSide(inst, args, filedes[0], new_jobs, new_jobs_lock,
            forked_downloads, forked_dl_lock,
            sigmask, orig_sigmask);

    if (us.start() != 0) {
        cerr << "Failed to start user interface" << endl;
        return 1;
    }

    int result = jm.start();

    inst->tofile();

    delete inst;
    delete new_jobs;
    delete forked_downloads;

    free(new_jobs_lock);
    free(forked_dl_lock);
    free(sigmask);
    free(orig_sigmask);
    free(args);

    return result;
}


