/* $Id: instrumentation.cpp,v 1.3 2007-11-08 15:43:40 aehyvari Exp $ */
/* start[./copyright.txt] */
/* Including data from file ./copyright.txt */
/* file opened */
/* Reading 775 bytes of data */
/* Start of include */
/*
 * Copyright 2007 Antti Hyvrinen
 *
 * This file is part of GridJM.
 *
 * GridJM is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * GridJM is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GridJM; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/* End of include */
/* stop[./copyright.txt] */

#include "instrumentation.h"
extern "C" {
    #include "parser/inslangutil.h"
}

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <iostream>
#include <sys/time.h>
#include <string.h>
#include <stdarg.h>

/**
 * Constructor of Instrumenter class
 */
Instrumenter::Instrumenter(string filename) {
    this->filename = filename;
    int fd = open(filename.c_str(),
        O_WRONLY | O_CREAT | O_TRUNC,
        S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH);
    if (fd == -1) {
        perror("open()");
        cerr << "Error opening output file " << filename << endl;
        return;
    }
    this->firstentry=-1;
    this->lastentry=-1;
    this->instlock = (pthread_mutex_t *) malloc(sizeof(pthread_mutex_t));
    pthread_mutex_init(this->instlock, 0);

    close(fd);
}

Instrumenter::~Instrumenter() {
    for (list<EvType*>::iterator ei = simevents.begin();
            ei != simevents.end(); ei++) {
        delete *ei;
    }
    free(this->instlock);
}

/**
 * Get current time
 */
int Instrumenter::GetTime() {
    struct timeval tv;
    struct timezone tz;
    tz.tz_minuteswest = 0;
    tz.tz_dsttime = 0;
    gettimeofday(&tv, &tz);
    return (int)((double)tv.tv_sec+(double)tv.tv_usec/1000000);
}


bool EvType::operator<=(EvType &rhs) {
    return (*this).start <= rhs.start ;
}

bool EvType::operator>=(EvType &rhs) {
    return rhs.start <= (*this).start;
}

bool EvType::operator>(EvType &rhs) {
    return not ((*this).start <= rhs.start);
}

bool EvType::operator<(EvType &rhs) {
    return (*this).start < rhs.start;
}

bool EvType::operator==(EvType &ev) {
    return (*this).start <= ev.start && ev.start <= (*this).start;
}

bool EvType::operator!=(EvType &ev) {
    return not ((*this).start == ev.start);
}
/**
 * Start an event for job named evname
 */
void Instrumenter::simstart(string evname) {
    EvType *e = new EvType();

    e->start = GetTime();
    e->name = evname;

    pthread_mutex_lock(instlock);
    {
        if (firstentry == -1)
            firstentry = e->start;
        lastentry = e->start;

        simevents.push_back(e);
    }
    pthread_mutex_unlock(instlock);

    return;
}



/**
 * Stop an event, with type type. Event must have been started
 */
void Instrumenter::simstop(string evname, event_t type) {
//    assert(! (simevents[evname] == NULL));

    EvType *event = NULL;

    pthread_mutex_lock(instlock);
    {
        for (list<EvType*>::iterator iter = simevents.begin();
                iter != simevents.end(); iter++) {
            if (((*iter)->name == evname) && (*iter)->stop == 0) {
                event = *iter;
                break;
            }
        }

        if (!event)
            assert(0); // Event not started

        event->stop = GetTime();
        if ((type == successfull) && event->isresubmitted())
            event->type = resubmit;
        else
            event->type = type;
        lastentry = event->stop;
    }
    pthread_mutex_unlock(instlock);

    return;
}

bool Instrumenter::evchange(string evname, event_t type) {

    bool found = false;
    int time = GetTime();

    pthread_mutex_lock(instlock);
    {
        // Browse through the event list to find the event
        for (list<EvType*>::iterator iter = simevents.begin();
            iter != simevents.end(); iter++)
            if (strcmp((*iter)->name.c_str(), evname.c_str()) == 0) {
                (*iter)->resubmitted = true; // A quick hack, sort of
                (*iter)->type = resubmit;
                found = true;
                break;
            }
    }
    pthread_mutex_unlock(instlock);

    // Event found
    return found;
}

/**
 * Read events from a file previously written by instrumenter
 */
int Instrumenter::readEvents(char * file) {
    int fd = open(file, O_RDONLY);
    if (fd == -1) {
        perror("open()");
        return -1;
    }

    struct stat s;

    if (stat(file, &s) != 0) {
        perror("stat()");
        return -1;
    }


    char *parsebuf = (char *) malloc(s.st_size + 1);


    if (read(fd, parsebuf, s.st_size) == -1) {
        perror("read()");
        return -1;
    }

    // Make sure it is null-terminated
    parsebuf[s.st_size] = '\0';

    if (close(fd) == -1) {
        perror("close()");
        return -1;
    }

    struct Node* p = parsestring(parsebuf);

    free(parsebuf);

    if (p != 0) {
        printf("Parse ok\n");

        if (p->children == NULL) {
            freeTree(p);
            return -3;
        }

        pthread_mutex_lock(instlock);
        {
            for (struct Node *first = p->children->first;
                    first != NULL;
                    first = first->next) {
                if ( strcmp(first->type, "simevents_t") == 0) {
                    for (struct Node *nl = first->children->first;
                            nl != NULL; nl = nl->next) {
                        struct NodeList* evlist = nl->children;
                        char *name   = evlist->first->value;
                        char *start = evlist->first->next->children->first->value;
                        char *stop  = evlist->first->next->children->last->value;
                        char *status = evlist->last->value;
                        EvType *e = new EvType();
                        e->start = (int)atof(start);
                        e->stop  = (int)atof(stop);
                        e->name  = name;
                        if (strcmp(status, "successfull") == 0)
                            e->type = successfull;
                        else if (strcmp(status, "resubmit") == 0)
                            e->type = resubmit;
                        else if (strcmp(status, "failed") == 0)
                            e->type = failed;
                        else if (strcmp(status, "undef") == 0)
                            e->type = undef;
                        else
                            e->type = undef;
                        simevents.push_back(e);
                    }
                }
            }
            freeTree(p);
        }
        pthread_mutex_unlock(instlock);

    }

    else {
        printf("Parse failed\n");
        return -2;
    }
    return 0;
}

/**
 * A helper for writing something to a buffer, taken from printf man
 * pages
 */
char* make_message(const char *fmt, ...)
{
    /* Guess we need no more than 100 bytes.  */
    int n, size = 100;
    char *p;
    va_list ap;
    if ((p = (char*) malloc (size)) == NULL)
        return NULL;

    while (1) {
        /* Try to print in the allocated space. */
        va_start(ap, fmt);
        n = vsnprintf(p, size, fmt, ap);
        va_end(ap);
        /* If that worked, return the string. */
        if (n > -1 && n < size)
            return p;
        /* Else try again with more space. */
        if (n > -1)
            /* glibc 2.1 */
            size = n+1;
        /* precisely what is needed */
        else
            /* glibc 2.0 */
            size *= 2;
        /* twice the old size */
        if ((p = (char*) realloc (p, size)) == NULL)
            return NULL;
    }
}

/**
 * Write the contents of the instrumentation to the output file
 */

void Instrumenter::tofile() {

    pthread_mutex_lock(instlock);
    {
        int output_fd = open(filename.c_str(),
            O_WRONLY | O_CREAT | O_TRUNC,
            S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH);
        if (output_fd == -1) {
            perror("open()");
            pthread_mutex_unlock(instlock);
            return;
        }

        int final_len = 0;
        char* header_buf = make_message("(\n(timescale\n (%d.0 %d.0)\n)\n(evtypes\n ()\n)\n(thnames\n ()\n)\n (jobnames\n ()\n)\n (runtimes (\n \n))\n (events (\n\
    ))\n (simeventsc (\n", firstentry, lastentry);

        final_len += strlen(header_buf);

        list<char*> simevents_strs;
        for (list<EvType*>::iterator iter = simevents.begin();
            iter != simevents.end(); iter++) {

            int stop_t = ((*iter)->stop == 0 ?
                    GetTime() :
                    (*iter)->stop);

            // To make parsing easy, we require names to be enclosed in
            // angle brackets.  If the name starts with one, it is assumed
            // to adhere to this protocol
            char *single_ev = make_message(" (%s%s%s\n  (%d.0 %d.0) %s)\n",
                ((*iter)->name[0] == '[' ? "" : "["),
                (*iter)->name.c_str(),
                ((*iter)->name[0] == '[' ? "" : "]"),
                (*iter)->start, stop_t,
                (*iter)->typeStr().c_str()
                );
            simevents_strs.push_back(single_ev);
            final_len += strlen(single_ev);
        }
        char *tail_buf = "))\n)\n";
        final_len += strlen(tail_buf);
        char *obuf = (char *)malloc(final_len+1);
        strcpy(obuf, header_buf);
        char* curr_ptr = &obuf[strlen(header_buf)];
        free(header_buf);
        for (list<char*>::iterator ci = simevents_strs.begin();
                ci != simevents_strs.end(); ci++) {
            strcpy(curr_ptr, *ci);
            curr_ptr += strlen(*ci);
            free(*ci);
        }
        strcpy(curr_ptr, tail_buf);

        ssize_t sz = write(output_fd, obuf, final_len);
        if (sz == -1) {
            perror("write()");
        }
        free(obuf);
        close(output_fd);
    }
    pthread_mutex_unlock(instlock);

    return;
}

/**
 * EvType declarations
 */
string EvType::typeStr() {
    if (type == successfull) return "successfull";
    if (type == resubmit) return "resubmit";
    if (type == failed) return "failed";
    return "undef";
}

bool EvType::isresubmitted() {
    return resubmitted;
}
