/* $Id: foo3.cpp,v 1.3 2007-09-14 14:21:26 aehyvari Exp $ */
/* start[./copyright.txt] */
/* stop[./copyright.txt] */

#include "SDL/SDL.h"
#include <string>

#include "../instrumentation.h"

const int SCREEN_WIDTH   = 640;
const int SCREEN_HEIGHT  = 480;
const int SCREEN_BPP     = 32;

SDL_Surface *message     = NULL;
SDL_Surface *background  = NULL;
SDL_Surface *screen      = NULL;

SDL_Event    event;

SDL_Surface *load_image( std::string filename )
{
    SDL_Surface *loadedImage = NULL;
    SDL_Surface *optimizedImage = NULL;

    loadedImage = SDL_LoadBMP( filename.c_str() );

    if ( loadedImage != NULL)
    {
        optimizedImage = SDL_DisplayFormat( loadedImage );
        SDL_FreeSurface( loadedImage );
    }

    return optimizedImage;
}

void apply_surface( int x, int y, SDL_Surface *source, SDL_Surface *destination )
{
    SDL_Rect offset;

    offset.x = x;
    offset.y = y;

    SDL_BlitSurface( source, NULL, destination, &offset );
}

void DrawRect( SDL_Surface *screen,
               Sint32 x1, Sint32 y1,
               Sint32 x2, Sint32 y2,
               Uint8 R, Uint8 G, Uint8 B,
               bool filled )
{
//    y1 = SCREEN_HEIGHT - y1;
//    y2 = SCREEN_HEIGHT - y2;
//    printf("rectangle %d %d %d %d\n", x1, y1, x2, y2);
    Uint32 color = SDL_MapRGB( screen->format, R, G, B );
    if ( SDL_MUSTLOCK(screen) ){
        if ( SDL_LockSurface(screen) < 0 ) {
            return;
        }
    }

    Uint32 *bufp;

    // Top hozirontal
    for ( Sint32 x = x1; x <= x2; x++ )
    {
        bufp = (Uint32 *)screen->pixels + y1*screen->pitch/4 + x;
        *bufp = color;
    }

    // Bottom horizontal
    for ( Sint32 x = x1; x <= x2; x++ )
    {
        bufp = (Uint32 *)screen->pixels + y2*screen->pitch/4 + x;
        *bufp = color;
    }

    // Left vertical
    for ( Sint32 y = y1; y <= y2; y++ )
    {
        bufp = (Uint32 *)screen->pixels + y*screen->pitch/4 + x1;
        *bufp = color;
    }

    // Right vertical
    for ( Sint32 y = y1; y <= y2; y++ )
    {
        bufp = (Uint32 *)screen->pixels + y*screen->pitch/4 + x2;
        *bufp = color;
    }

    // If requested to fill, draw the inside as well
    if ( filled )
    {
        for ( Sint32 x = x1+1; x < x2; x++ )
        {
            for ( Sint32 y = y1 + 1; y < y2; y++ )
            {
                bufp = (Uint32 *)screen->pixels + y*screen->pitch/4 + x;
//                Uint8 Rp, Gp, Bp;
//                SDL_GetRGB(*bufp, screen->format, &Rp, &Gp, &Bp);
//                color = SDL_MapRGB(screen->format, (R+Rp)<<2, (G+Gp)<<2, (B+Bp)<<2);
                *bufp = color;
            }
        }
    }

    if ( SDL_MUSTLOCK( screen ) )
        SDL_UnlockSurface(screen);

//    SDL_UpdateRect(screen, x, y, 1, 1);
}

class Track
{
    public:
        list<EvType*> events;
        void toString();
};

void Track::toString()
{
    for (list<EvType*>::iterator ei = events.begin();
        ei != events.end();
        ei++)
    {
        printf("  %d %d %s %s\n",
            (*ei)->start,
            (*ei)->stop,
            (*ei)->name.c_str(),
            (*ei)->typeStr().c_str());
    }
}

class Overhead
{
    public:
        ~Overhead();
        int     mintime;           // Smallest time
        int     maxtime;           // Largest time
        double  xscale() { return SCREEN_WIDTH / (double)(maxtime-mintime); }
        int     height();          // in pixels, the height of a timeline entry
        int     skip();            // in pixels, the distance between two tracks
        list<Track*> tracks;   // List of timeline-elements
        list<EvType*> events;  // all events, to be placed on tracks
        void    toString();
        void    drawOverhead(SDL_Surface *);
        void    drawEvt(EvType *evt, int track);
        SDL_Surface *screen;
};

Overhead::~Overhead() {
    for (list<Track*>::iterator ti = tracks.begin();
            ti != tracks.end(); ti++) {
        delete *ti;
    }
}

int Overhead::height() {
    return (int) (3/((double)4)*((double)SCREEN_HEIGHT)/((double)tracks.size()));
}

int Overhead::skip() {
    return (int) (1/((double)4)*((double)SCREEN_HEIGHT)/((double)tracks.size()));
}

void Overhead::toString()
{
    printf("Interval:\n %d %d\n", mintime, maxtime);
    printf("Tracks (%d):\n", tracks.size());
    int i = 0;
//    for (list<Track*>::iterator ti = tracks.begin();
//        ti != tracks.end();
//        ti++, i++)
//    {
//        printf(" Track %d:\n", i);
//        (*ti)->toString();
//    }
}

void Overhead::drawOverhead( SDL_Surface *screen )
{
    this->screen        = screen;
    int current_t       = 0;

    for (list<Track*>::iterator ti = tracks.begin();
        ti != tracks.end();
        ti++, current_t++)
    {
        Track *ct = *ti;
        for (list<EvType*>::iterator ei = ct->events.begin();
            ei != ct->events.end();
            ei ++)
        {
            drawEvt(*ei, current_t);
        }
    }
}
void Overhead::drawEvt( EvType *evt, int track )
{

    Uint8 fr, fg, fb;
    switch (evt->type) {
        case successfull:
        {
            fr = 0;
            fg = (Uint8)128;
            fb = 0;
            break;
        }
        case resubmit:
        {
            fr = (Uint8)178;
            fg = (Uint8)178;
            fb = 0;
            break;
        }
        case failed:
        {
            fr = (Uint8)128;
            fg = 0;
            fb = 0;
            break;
        }
        case submitfail:
        {
            fr = (Uint8)255;
            fg = 0;
            fb = 0;
            break;
        }
        case undef:
        {
            fr = 0;
            fg = 0;
            fb = (Uint8)255;
            break;
        }
    }

    /* This is the colouring of the box */
    DrawRect(
        screen,
        (int)((evt->start-mintime)*xscale()),
        track*height() + skip(),
        (int)((evt->stop-mintime)*xscale()),
        (track+1)*height(),
        fr, fg, fb, true
    );
    /* Here we draw the cpu time */
//    DrawRect( screen, 170, 150, 230, 170, 128, 128, 128, true );
    /* These are the borders */
    DrawRect(
        screen,
        (int)((evt->start-mintime)*xscale()),
        track*height() + skip(),
        (int)((evt->stop-mintime)*xscale()),
        (track+1)*(height()),
        0, 0, 0, false
    );

    return;
}

int appendToTrack(EvType *evt, Overhead *oh)
{
    bool processed = false;

    for (list<Track*>::iterator ti = oh->tracks.begin();
        ti != oh->tracks.end();
        ti++)
    {
        EvType *lastel = (*ti)->events.back();
        if (evt->start >= lastel->stop) {
            (*ti)->events.push_back(evt);
            processed = true;
            break;
        }
    }
    if (not processed)
    {
        Track* ntrack = new Track();
        oh->tracks.push_back(ntrack);
        ntrack->events.push_back(evt);
    }
}

int main ( int argc, char **argv )
{
    if (argc != 2 ) {
        printf("Usage: %s gridjm.ins\n", argv[0]);
        return 1;
    }

    if ( SDL_Init( SDL_INIT_EVERYTHING )  == -1 )
        return 1;

    screen = SDL_SetVideoMode( SCREEN_WIDTH, SCREEN_HEIGHT, SCREEN_BPP, SDL_SWSURFACE );

    if ( screen == NULL )
        return 1;

    SDL_WM_SetCaption( "GridJM Visualize", "GridJM Visualize" );

    message = load_image( "hello_world.bmp" );
    background = load_image( "background.bmp" );
    apply_surface( 0, 0, background, screen );

    apply_surface( 180, 140, message, screen );

    if ( SDL_Flip( screen ) == -1 )
        return 1;

    bool quit = false;

    while (quit == false) {

        while (SDL_PollEvent( &event )) {

            if ( event.type == SDL_QUIT )
                quit = true;

            Instrumenter *inst = new Instrumenter("/tmp/foo.ins");

            if (inst->readEvents(argv[1]) == 0) {
                apply_surface( 0, 0, background, screen );

                list<EvType> sortedlist;
                for (list<EvType*>::iterator evt = inst->simevents.begin();
                        evt != inst->simevents.end();
                        evt++)
                {
    //                printf("%s: %d %d\n", (*evt)->name.c_str(), (*evt)->start, (*evt)->stop);
                    sortedlist.push_back(**evt);
                }


                sortedlist.sort();

                // Start to construct the overhead table
                Overhead *oh = new Overhead();

                // Minimum time is easy
                oh->mintime = sortedlist.front().start;
                // This is our initial guess
                oh->maxtime = oh->mintime;
                for (list<EvType>::iterator evt = sortedlist.begin();
                        evt != sortedlist.end();
                        evt++)
                {
            //        printf("%s: %d %d\n", (*evt).name.c_str(), (*evt).start, (*evt).stop);
                    // Update max time
                    oh->maxtime = (*evt).stop > oh->maxtime ? (*evt).stop : oh->maxtime;
                    appendToTrack(&(*evt), oh);

                }

                oh->toString();

                oh->drawOverhead(screen);

                delete oh;

                if ( SDL_Flip( screen ) == -1 )
                    return 1;
            }

            delete inst;

            SDL_Delay( 4000 );
        }
    }
    SDL_FreeSurface( message );
    SDL_FreeSurface( background );

    SDL_Quit();

    return 0;
}
