#ifndef ARCLIB_JOBSUBMISSION_BQ
#define ARCLIB_JOBSUBMISSION_BQ

#ifndef VERSION
#define VERSION "0.6.0"
#endif

#include <list>

#include <arc/common.h>
#include <arc/error.h>
#include <arc/giis.h>
#include <arc/mdsquery.h>
#include <arc/target.h>
#include <arc/xrsl.h>
#include <arc/jobsubmission.h>

#include "queuelist.h"

/** Class taking care of JobSubmission after brokering. To submit a job,
 *  pass the xrsl and the list of surviving targets to the constructor and
 *  call Submit(). Submit() takes care of calling PrepareXrsl() that prepares
 *  the xrsl before jobsubmission.
 */
class JobSubmission_bq {
	public:
		/** The constructor takes the list of possible targets and the
		 *  original xrsl as input-parameters.
		 */
		JobSubmission_bq(Xrsl axrsl,
		              std::list<Target> targetlist,
		              bool dryrun = false);

		/** This function performs JobSubmission using JobFTPControl to the
		 *  specified target.
		 */
		std::string Submit(int timeout = TIMEOUT)
		throw(XrslError, JobSubmissionError);

		/** This function prepares the xrsl for JobSubmission to the chosen
		 *  target. It rewrites the xrsl so that it's ready to be submitted.
		 */
		Xrsl PrepareXrsl(Target& chosentarget)
		throw(XrslError, JobSubmissionError);

		/** This method register the job-submission in the original list
		 *  of queues. This method should be called after submitting one
		 *  job of a multi-job xrsl and before brokering is performed on the
		 *  next xrsl.
		 */
		void RegisterJobsubmission(std::list<Queue>& queues);

		/**
		 * A list of bad queues, constructed during the
		 * submission phase, when a submission to a queue fails
		 * (usually because of a timeout, so that the knowledge
		 * is expensive to gather).  This information can be
		 * collected after a submission and used when
		 * resubmitting
		 */
		std::list<QueueName> badQueues;
	private:
		std::list<Target> targets;
		Xrsl xrsl;
		bool dryrun;

		std::multimap<std::string, std::string> localinputfiles;

		/** properties of last submission-target */
		std::list<Target>::iterator jobtarget;
		int jobcount;
		int jobcputime;
		long long jobdisk;

		/** This method ensures that the attribute attr is represented in
		 *  the double-value list inoutattr. Otherwise it is added. This is
		 *  typically needed for the inputfiles and outputfiles attributes.
		 */
		void PrepareInputOutput(Xrsl& jobxrsl,
		                        const std::string& attr,
		                        const std::string& inoutattr);

		/** This method rewrites the inputfiles attribute. */
		void PrepareUpload(Xrsl& jobxrsl) throw(XrslError);

		/** This method rewrites the middleware, runtimeenvironment and
		 *  opsys attributes.
		 */
		void PrepareRE(Xrsl& jobxrsl,
		               const std::string& attr,
		               Target& chosentarget,
		               bool oldserver) throw(XrslError);
};

/** This method submits a job specified by the xrsl. */
std::string SubmitJob_bq(Xrsl axrsl,
                      std::list<Target> targets,
                      int timeout = TIMEOUT,
                      bool dryrun = false) throw(JobSubmissionError, XrslError);

#endif // ARCLIB_JOBSUBMISSION_BQ
