/* $Id: userside.cpp,v 1.3 2007/04/10 09:07:48 aehyvari Exp $ */
/* Copyright 2003 Henrik Thostrup Jensen and Jesper Ryge Leth
 * All rights reserved.
 *
 * This file is part of NGProxy.
 *
 * NGProxy is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * NGProxy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NGProxy; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* start[./copyright.txt] */
/* Including data from file ./copyright.txt */
/* file opened */
/* Reading 775 bytes of data */
/* Start of include */
/*
 * Copyright 2007 Antti Hyvrinen
 *
 * This file is part of GridJM.
 *
 * GridJM is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * GridJM is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GridJM; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/* End of include */
/* stop[./copyright.txt] */

/* user side communication module implementation */

#include <iostream>
#include <sys/socket.h>
#include <sys/types.h>
#include <assert.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/wait.h>
#include <errno.h>

#include "userside.h"
#include "parseargs.h"

void *user_listener(void *pointer) {

    UserSide *us = (UserSide*) pointer;
    // Start listening
    int retval = 0;
    if (us->args->useuds) {
        cout << "Using uds for communication" << endl;
        retval = us->Listen(true);
    } else {
        cout << "Using net for communication" << endl;
        retval = us->Listen(false);
    }
    // Exit thread (we only get here on failure)
    pthread_exit((void *) retval);
}


UserSide::UserSide(struct arguments *a, int pipe_read,
        queue<char*> *new_jobs, pthread_mutex_t *new_jobs_lock,
        list<PidJob*> *forked_downloads, pthread_mutex_t *forked_dl_lock,
        sigset_t *sigmask, sigset_t *orig_sigmask) {
    this->args = a;
    this->pipe_read = pipe_read;
    this->new_jobs = new_jobs;
    this->new_jobs_lock = new_jobs_lock;
    this->forked_downloads = forked_downloads;
    this->forked_dl_lock = forked_dl_lock;
    this->sigmask = sigmask;
    this->orig_sigmask = orig_sigmask;
}

int UserSide::start() {
    pthread_t ui_thread;
    int retval = pthread_create(&ui_thread, 0, user_listener, this);
    if (retval != 0) {
        cerr << "Error creating user listener thread, exiting" << endl;
    }
    return retval;
}

int UserSide::Listen(bool uds) {

    // Create socket
    int server_socket;
    if (uds)
        server_socket = socket(AF_UNIX, SOCK_STREAM, 0);
    else
        server_socket = socket(PF_INET, SOCK_STREAM,0);

    if (server_socket == -1) {
        int tmperr = errno;
        char *errdesc = (char *)malloc(1024);
        cerr << "socket() " << strerror_r(tmperr, errdesc, 1024) << endl;
        free(errdesc);
        return -1;
    }

    int on = 1;
    setsockopt(server_socket, SOL_SOCKET,
            SO_REUSEADDR, (void*)(&on), sizeof(on));

    // We need both declarations here
    struct sockaddr_un uds_server;
    struct sockaddr_in in_server;
    struct sockaddr *server; /* Later the correct will be here */
    struct in_addr sin_addr;

    // Create socket address struct used for binding
    if (uds) {
        uds_server.sun_family = AF_UNIX;
        strcpy(uds_server.sun_path, args->socketstring);

        // Delete any previous file
        unlink(args->socketstring);
    }
    else {
        inet_aton("0.0.0.0", &sin_addr);

        in_server.sin_family = AF_INET;
        in_server.sin_port = htons(args->port);
        in_server.sin_addr = sin_addr;
    }

    int retval = -1; // used for return values

    if (uds)
        server = (struct sockaddr*) &uds_server;
    else
        server = (struct sockaddr*) &in_server;

    if (uds) {
        // Bind socket to file
        retval = bind(server_socket, server, sizeof(struct sockaddr_un));
        if (retval == -1) {
            int tmperr = errno;
            char *errdesc = (char *)malloc(1024);
            cerr << strerror_r(tmperr, errdesc, 1024) << endl;
            free(errdesc);
            return -1;
        }
    }
    else {
        /* Retry if necessary, until we have a connection */
        while (retval == -1) {
            retval = bind(server_socket, server, sizeof(struct sockaddr_in));
            if (retval == -1) {
                int tmperr = errno;
                char *errdesc = (char *)malloc(1024);
                cerr << "bind() " << strerror_r(tmperr, errdesc, 1024) << endl;
                free(errdesc);
                sleep(1);
            }
        }
    }

    // Start listening
    retval = listen(server_socket,2);
    if (retval == -1) {
        int tmperr = errno;
        char *errdesc = (char *)malloc(1024);
        cerr << "listen() " << strerror_r(tmperr, errdesc, 1024) << endl;
        free(errdesc);
        return -1;
    }
    cout << "Listener thread started" << endl;

    struct sockaddr_un uds_client;
    struct sockaddr_in in_client;
    struct sockaddr *client;
    int addrlen;

    if (uds) {
        addrlen = sizeof(struct sockaddr_un);
        client = (struct sockaddr *)&uds_client;
    }
    else {
        addrlen = sizeof(struct sockaddr_in);
        client = (struct sockaddr *)&in_client;
    }

    retval = 0;
    // Start accepting connections
    while (retval == 0) {
        // Wait for connection
        int socket_handle = accept(server_socket, client,
                                   (socklen_t*) &addrlen);
        if (socket_handle == -1) {
            int tmperr = errno;
            char *errdesc = (char *)malloc(1024);
            cerr << "accept() " << strerror_r(tmperr, errdesc, 1024) << endl;
            free(errdesc);
            retval = -1;
        }
        else {
            cout << "Accepted Connection" << endl;
            // If we want to respond faster to multiple connections we 
            // could create a thread here
            HandleConnection(socket_handle);
            socket_handle = -1;
        }
    }
    return retval;
}

/*
 * Do all socket communication with the user from here, and in addition
 * listen to the exiting download processes.
 *
 * Return 0 if the user disconnected
 *       -1 if other error occurred
 */
int UserSide::HandleConnection(int socket_handle) {

    int returnvalue = 0;

    const int MAX_BUFFER = 1024;
    char buffer[MAX_BUFFER];
    // EOF needs to be cleared so we need a sockstream because of
    // the interface of clearerr.
    FILE *sockstream = fdopen(socket_handle, "rw");
    // Could be that the fileno changes at this point
    socket_handle = fileno(sockstream);
    /* We finally get the xrsl socket after this */
    xrslsocket = socket_handle;

    fd_set *rfds = (fd_set *) malloc(sizeof(fd_set));

    /* Main loop: Start listening to events. */
    bool ok = 1;
    int retval;
    while (ok) {

        /* Watch socket. The values have to be reset after a
         * (possible) signal */
        FD_ZERO(rfds);
        FD_SET(socket_handle, rfds);
        if (pipe_read != -1)
            FD_SET(pipe_read, rfds);

        retval = pselect(max(socket_handle, pipe_read) + 1,
                rfds, NULL, NULL, NULL, orig_sigmask);

        /* Got a signal, or error in select */
        if (retval == -1) {
            if (errno == EINTR) {
                /* A signal got received */
                cout << "Finished download" << endl;
                WaitPids();
            }
            else {
                int tmperr = errno;
                char *errdesc = (char *)malloc(1024);
                cerr << "pselect() " << strerror_r(tmperr, errdesc, 1024) << endl;
                ok = false;
                returnvalue = -1;
                free(errdesc);
            }
        }

        /* Got data from input */
        if (retval > 0) {
            if (FD_ISSET(socket_handle, rfds)) {
                cout << "User sends data" << endl;
                int bufsize = BUFSIZE;
                int curr = 0;
                int count = 0;

                char *buf = (char *) malloc(bufsize);

                int read_still = 1;
                while (read_still) {
                    while (READSIZE + curr > bufsize) {
                        buf = (char *) realloc(buf, bufsize * 2);
                        bufsize = 2*bufsize;
                    }
                    count = read(socket_handle, buf+curr, READSIZE);
                    if (count < READSIZE)
                        read_still = 0;
                    curr += count;
                }

                if (count < 0) {
                    int tmperr = errno;
                    char *errdesc = (char *)malloc(1024);
                    cerr << "socket read() " << strerror_r(tmperr, errdesc, 1024) << endl;
                    ok = false;
                    free(errdesc);
                    free(buf);
                    returnvalue = -1;
                }

                else if (count == 0) {
                    /* This check is needed so that if the user disconnects,
                     * we do not hang forever reading null, but instead can
                     * accept a new connection */
                    cout << "Read EOF" << endl;
                    clearerr(sockstream);
                    read_still = 0;
                    free(buf);
                    ok = 0;
                }

                else {
                    /* Normal reading */
                    assert(curr >= 1); /* If we read directly EOF */
                    if (buf[curr-1] == EOF) {
                        cerr << "Received EOF" << endl;
                        read_still = 0;
                        /* EOF is the end delimiter. We can still be on this
                         * loop, so keep on the godo work */
                        clearerr(sockstream);
                        buf[curr-1] = '\0';
                        InsertXrsl(buf); /* New xrsl ready */
                        /* Destruction of buf is handled elsewhere after
                         * this, that is, by the function converting it to
                         * an xrsl */
                    }
                }
            }
            /* Request for new jobs */
            if (FD_ISSET(pipe_read, rfds)) {
                cout << "GridJM sends data" << endl;
                int bufsize = BUFSIZE;
                int curr = 0;
                int count = 0;

                char *buf = (char *) malloc(bufsize);

                int read_still = 1;
                while (read_still) {
                    while (READSIZE + curr > bufsize) {
                        buf = (char *) realloc(buf, bufsize * 2);
                        bufsize = 2*bufsize;
                    }
                    count = read(pipe_read, buf+curr, READSIZE);
                    if (count < READSIZE)
                        read_still = 0;
                    curr += count;
                }

                if (count < 0) {
                    int tmperr = errno;
                    char *errdesc = (char *)malloc(1024);
                    cerr << "pipe read() " << strerror_r(tmperr, errdesc, 1024) << endl;
                    ok = false;
                    free(errdesc);
                    free(buf);
                    returnvalue = -1;
                }

                if (count == 0) {
                    /* This should not happen, unless the gridjm thread
                     * crashes. Set pipe_read to -1 and continue,
                     * althoug no new requests will be received then.
                     */
                    cout << "Pipe read EOF!" << endl;
                    read_still = 0;
                    free(buf);
                    pipe_read = -1;
                }

                else {
                    /* Normal reading */
                    assert(curr >= 1);
                    if (buf[curr-1] == '\n') {
                        buf = (char *)realloc(buf, curr+1);
                        read_still = 0;
                        /* '\n' is the end delimiter. */
                        buf[curr] = '\0';
                        RequestJob(buf); /* New job requests */
                        free(buf);
                    }
                }
            }
        }
    }
    free(rfds);
    return returnvalue;
}

/*
 * Wait for all pids listed in forked_downloads and report the finishing
 * to the user. First part of the method holds the forked_dl_lock.
 * Before reporting, the lock is released. FIXME Make finished_downloads
 * a class variable, so that it is preserved over client disconnects.
 */

void UserSide::WaitPids() {

    list<string> finished_downloads;

    pthread_mutex_lock(forked_dl_lock);
    int size = forked_downloads->size();
    list<PidJob*>::iterator cpid = forked_downloads->begin();

    while (cpid != forked_downloads->end()) {
        int status;
        int rval = waitpid((*cpid)->pid, &status, WNOHANG); // try to clean
        if (rval > 0) {
            PidJob *cpid_ptr = *cpid;
            forked_downloads->erase(cpid++);
            // Put this info on a list and pass it to the reader.
//            finished_downloads.push_back(cpid_ptr->name);
            // This is a bit of a hack, but we get the last component
            // which identifies the download directory
            finished_downloads.push_back(basename(cpid_ptr->name.c_str()));
            delete cpid_ptr;
        }
        else {
            cpid++;
        }
    }
    pthread_mutex_unlock(forked_dl_lock);

    // If socket is up, report the finished
    if (xrslsocket != -1) {
        list<string>::iterator jname = finished_downloads.begin();
        while (jname != finished_downloads.end()) {
            char *output;
            int dlsize =
                 asprintf(&output, "# dlfinish %s/%s\n",
                         args->dir, (*jname).c_str());
            write(xrslsocket, output, dlsize);
            free(output);
            jname++;
        }
    }
}

/*
 * Send request to the user socket. We could do some more elaborate work
 * here, such as shutting down the thread at request...
 */
void UserSide::RequestJob(char *buf) {
    if (xrslsocket != -1)
        write(xrslsocket, buf, strlen(buf));
}


/*
 * To avoid using the (possibly thread unsafe) nordugrid routines form
 * two different threads, we pass the xrsl as a char*
 */
void UserSide::InsertXrsl(char *buf) {

    // Grap xrsl vector lock, push the Xrsl object into
    // it, and realease lock
    pthread_mutex_lock(new_jobs_lock);
    new_jobs->push(buf);
    pthread_mutex_unlock(new_jobs_lock);

    cout << "New job was put into queue" << endl;
}
