#!/usr/bin/env python
"""Two occasions:
(1) Given a tar.gz file, extract the satjm
    instrumentation and grid results and give statistics of grid job
    submission overhead.
(2) Given a directory of grid results and a satjm instrumentation file,
    give the statistics of grid job submission overhead."""

import inssem
import sys
import os
import griddlparser
from keyquicksort import sortbykey
from griddlparser import NoOutputError

class Tmpl:
    pass

class Overhead:
    def __init__(self, dir, inst):

        self.height = 30 # In dots, the height of a timeline entry in ps
        self.skip = 10 # In dots, the distance between two tracks in ps

        self.timeline = [] # List of timeline-elements (which are lists)
        self.tracks = [] # list of the tracks containing same elements

        self.ins = inssem.construct1(inst)
        self.grids = {}
        dirs = os.listdir(dir)
        dirs = map(lambda x: os.path.join(dir, x), dirs)
        for d in dirs:

            try:
                gridp = griddlparser.Construct(d)
            except NoOutputError, e:
                print "no output on %s: %s" % (d, e)
                continue # Just skip this

            self.grids[gridp.jobname] = gridp

        combevents = {}
        for ev in self.ins.simeventsc:
            evduration = self.ins.simeventsc[ev]
            try:
                gridp = self.grids[ev]
            except KeyError, e:
                # The run time is not downloaded
                gridp = Tmpl
                gridp.runtime = -1

            # Append the element of timeline
            self.timeline.append([ev, evduration.t0, evduration.t1,
                evduration.type, gridp.runtime])

        sortbykey(self.timeline, 1)

        # The timeline elements use the same resource if the first stops
        # and the second starts within 10 seconds.
        prev = []
        for el in self.timeline:
            self.appendtotrack(el)

        self.mintime = min(map(min,
            map(lambda track: map(lambda el: el[1], track), self.tracks)))

        self.maxtime = max(map(max,
            map(lambda track: map(lambda el: el[2], track),
                self.tracks)))

    def appendtotrack(self, el):
        """Find the first free track and append the element to it"""

        processed = False

        for track in self.tracks:
            # Compare to last elements finish time and new elements
            # start time
            last = track[len(track)-1]
            if (el[1] - last[2]) >= -10:
                track.append(el)
                processed = True
                return

        if not processed:
            self.tracks.append([el])

    def elttops(self, el, x, y):
        height = self.height
        type = "undef"
        if el[3] == "successfull":
            color = "0.0 1.0 0.0"
        elif el[3] == "failed":
            color = "1.0 0.0 0.0"
        elif el[3] == "resubmit":
            color = ".7 .7 0.0"
        else:
            color = "0.0 0.0 1.0"

        fullbox = "\n".join([\
                    "% Full Box", \
                    color,
                    " setrgbcolor", \
                    "newpath", \
                    str(x) + " " + str(y) + " moveto", \
                    "0 " + str(height) + " rlineto", \
                    str(el[2] - el[1]) + " 0 rlineto", \
                    "0 " + str(-height) + " rlineto", \
                    "closepath fill\n"])

        # The outline
        outbox = "\n".join([\
                    "% Full Box", \
                    "0.0 setgray", \
                    "newpath", \
                    str(x) + " " + str(y) + " moveto", \
                    "0 " + str(height) + " rlineto", \
                    str(el[2] - el[1]) + " 0 rlineto", \
                    "0 " + str(-height) + " rlineto", \
                    "closepath stroke\n"])

        if el[4] >= 0:
            count = (el[2]-el[1])/2 - el[4]/2
            effbox = "\n".join([\
                "% Effective box", \
                "0.7 setgray", \
                "newpath", \
                str(count + x) + " " + str(y) + " moveto", \
                "0 " + str(height) + " rlineto", \
                str(el[4]) + " 0 rlineto", \
                "0 " + str(-height) + " rlineto", \
                "closepath fill\n"])
        else:
            effbox = ""

        return "\n".join([fullbox, effbox, outbox])

    def tracktops(self, trackno):
        pstracks = []
        track = self.tracks[trackno]

        bot = trackno * (self.height + self.skip)

        for el in track:
            left = el[1] - self.mintime
            psel = self.elttops(el, left, bot)
            pstracks.append(psel)

        return "\n".join(pstracks)

    def tops(self, psfile):
        ps = open(psfile, 'w')
        ps.write("%!\n")
        xscale = (7.8*72)/(self.maxtime - self.mintime)
        yscale = (11.0*72)/(len(self.tracks)*(self.skip + self.height))
        xlift = 0.2*72
        ylift = 0.5*72
        ps.write("%%BoundingBox: 0 0 562 792\n")
        ps.write(str(xlift) + " " + str(ylift) + " translate\n")
        ps.write(str(xscale) + " " + str(yscale) + " scale\n")
        for i in range(0, len(self.tracks)):
            ps.write(self.tracktops(i))

        # Here we print the scale to the bottom
        height = self.height
        # scaleheight = height/5
        scaleheight = len(self.tracks)*(self.skip + self.height)
        ref = "0 setgray\nnewpath\n" + "0 " + str(-height) + " moveto\n"
        for i in range(int(self.mintime), int(self.maxtime), 60*60):
            ref = "".join([ref, "0 ", str(scaleheight), " rlineto\n"])
            ref = "".join([ref, "0 ", str(-scaleheight), " rlineto\n"])
            ref = "".join([ref, str(60*60), " 0 rlineto\n"])
        ref = "".join([ref, "0 ", str(scaleheight), " rlineto\n"])
        ref = "".join([ref, "0 ", str(-scaleheight), " rlineto\n"])
        ref = "".join([ref, "closepath stroke\n"])

        ps.write(ref)

        ps.write("showpage\n")
        ps.close()

    def conclude(self):
        """Print the conclusions for each track"""

        conclusions = []
        for track in self.tracks:
            cpu = 0
            inuse = 0

            for el in track:
                if el[4] >= 0:
                    cpu += el[4]
                inuse += el[2] - el[1]
            conclusions.append([cpu, inuse])

        mintime = self.mintime
        maxtime = self.maxtime

        i = 0
        for c in conclusions:
            print "Track", i
            print "  total cpu", c[0]
            print "  total in use", c[1]
            print "  utilization", c[1]/(maxtime-mintime)
            print "  effective utilization", c[0]/(maxtime-mintime)
            i += 1

    def __str__(self):
        i = 0
        trs = []
        for track in self.tracks:
            trs.append("\n".join(["Track " + str(i),
                "\n".join(map(str, track))]))
            i += 1

        return "\n".join(trs)


if __name__ == '__main__':
    if len(sys.argv) < 3:
        print "Usage:", sys.argv[0], "griddir satjm.ins [ps-file]"
        sys.exit(1)
    else:
        oh = Overhead(sys.argv[1], sys.argv[2])
        print oh
        oh.conclude()
        if len(sys.argv) == 4:
            oh.tops(sys.argv[3])
        else:
            oh.tops("/tmp/foo.ps")
else:
    print "Parse by running", __name__ + ".Overhead(griddir, ins)"

