#include "instrumentation.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <iostream>
#include <sys/time.h>
#include <string.h>
/**
 * Constructor of Instrumenter class
 */
Instrumenter::Instrumenter(string filename) {
    this->filename = filename;
    output.open(filename.c_str());
    if (!output) {
        cerr << "Error opening output file" << endl;
        return;
    }
    this->firstentry=-1;
    this->lastentry=-1;
    output.close();
}

Instrumenter::~Instrumenter() {

}

/**
 * Get current time
 */
int Instrumenter::GetTime() {
    struct timeval tv;
    struct timezone tz;
    tz.tz_minuteswest = 0;
    tz.tz_dsttime = 0;
    gettimeofday(&tv, &tz);
    return (int)((double)tv.tv_sec+(double)tv.tv_usec/1000000);
}

/**
 * Start an event for job named evname
 */
void Instrumenter::simstart(string evname) {
    EvType *e = new EvType();

    e->start = GetTime();
    e->name = evname;

    if (firstentry == -1)
        firstentry = e->start;
    lastentry = e->start;

    simevents.push_back(e);
    return;
}



/**
 * Stop an event, with type type. Event must have been started
 */
void Instrumenter::simstop(string evname, event_t type) {
//    assert(! (simevents[evname] == NULL));

    EvType *event = NULL;
    for (list<EvType*>::iterator iter = simevents.begin();
            iter != simevents.end(); iter++) {
        if (((*iter)->name == evname) && (*iter)->stop == 0) {
            event = *iter;
            break;
        }
    }

    if (!event)
        assert(0); // Event not started

    event->stop = GetTime();
    if ((type == successfull) && event->isresubmitted())
        event->type = resubmit;
    else
        event->type = type;
    lastentry = event->stop;
    return;
}

bool Instrumenter::evchange(string evname, event_t type) {

    bool found = false;
    int time = GetTime();
    // Browse through the event list to find the event
    for (list<EvType*>::iterator iter = simevents.begin();
        iter != simevents.end(); iter++)
        if (strcmp((*iter)->name.c_str(), evname.c_str()) == 0) {
            (*iter)->resubmitted = true; // A quick hack, sort of
            (*iter)->type = resubmit;
            found = true; break;
        }
    // Event found
    return found;
}

/**
 * Write the contents of the instrumentation to the output file
 */

void Instrumenter::tofile() {
    output.open(filename.c_str());
    output << "(\n(timescale\n (" << firstentry << ".0 " <<
        lastentry << ".0)\n)\n";

    output << "(evtypes\n ()\n)\n";

    output << "(thnames\n ()\n)\n";

    output << "(jobnames\n ()\n)\n";

    output << "(runtimes (\n \n))\n";

    output << "(events (\n))\n";

    output << "(simeventsc (\n";
    for (list<EvType*>::iterator iter = simevents.begin();
        iter != simevents.end(); iter++) {

        int stop_t = ((*iter)->stop == 0 ?
                GetTime() :
                (*iter)->stop);

        output << " (" << (*iter)->name << "\n  (" << \
                      (*iter)->start << ".0 " <<
                      stop_t << ".0) " << (*iter)->typeStr() << ")\n";
    }
    output << "))\n)\n";
    output.flush();
    output.close();
    return;
}

/**
 * EvType declarations
 */
string EvType::typeStr() {
    if (type == successfull) return "successfull";
    if (type == resubmit) return "resubmit";
    if (type == failed) return "failed";
    return "undef";
}

bool EvType::isresubmitted() {
    return resubmitted;
}
