#include <list>
#include <queue>
#include <iostream>
#include <arc/mdsdiscovery.h>
#include <arc/mdsquery.h>
#include <arc/standardbrokers.h>
#include <arc/jobsubmission.h>
#include <arc/jobftpcontrol.h>
#include <arc/joblist.h>
#include <arc/url.h>
#include <arc/target.h>
#include <arc/notify.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <signal.h>
#include "gridjm.h"
#include "parseargs.h"

GridJM::GridJM(Instrumenter* i, struct arguments * a, int p_write,
                queue<char*> *n_jobs, pthread_mutex_t *n_jobs_lock,
                list<PidJob*> *f_downloads, pthread_mutex_t *f_dl_lock) {
    args = a;                   /* Arguments */
    num_current_jobs = 0;       /* Jobs currently in grid */
    clupd = 0;                  /* Cluster list last updated */
    qupd = 0;                   /* Queue list last updated */
    inst = i;                   /* Instrumentation module */
    pipe_write = p_write;       /* Ask for more jobs here */
    new_jobs = n_jobs;          /* New job queue */
    new_jobs_lock = n_jobs_lock; /* Lock for new job queue */
    forked_downloads = f_downloads; /* Forked downloads (pid & name) */
    forked_dl_lock = f_dl_lock; /* Lock for forked downloads */
}

int GridJM::start() {
    globus_module_activate(GLOBUS_XIO_MODULE);
    return mainloop();
}

int GridJM::mainloop() {

    int last_time = time(0);
    int time_left = 0;

    do {
        /* Clean the forked downloads & report the finishing
         * Check if it is time to update the job statuses */
        if (time(0) - last_time > args->interval) {
            UpdateJobStatus();
            last_time = time(0);
        }

        /* Update the available clusters if list is old */
        if ((clusters.size() == 0) || (time(0) - clupd > args->clupdp)) {
            try {
                clusters = GetClusterResources();
                clupd = time(0);
            } catch (MDSDiscoveryError e) {
                notify(WARNING) << "Error: " << e.what() << endl;
            }
        }

        /* Update the available queues if list is old */
        if ((queuelist.size() == 0) || (time(0) - qupd > args->qupdp)) {
            try {
                queuelist = GetQueueInfo(clusters,
                        MDS_FILTER_CLUSTERINFO, true, "", 20);
                qupd = time(0);
            } catch (MDSDiscoveryError e) {
                notify(WARNING) << "Error: " << e.what() << endl;
            }
        }

        /* Check the new job queue */
        HandleNewJobs();

        time_left = sleep(1);
        if (time_left) {
            // If we get here, we where interrupted during sleep
            notify(INFO) << "The sleeper awakens" << endl;
        }
    } while (running);

    /* FIXME some sensible error reporting, please */
    return -1;
}

void GridJM::HandleNewJobs() {
    /**
     * We now check for the arrival of new jobs
     * This is done by acquiring lock, see if there is anything
     * in the new_jobs vector, we take one out, release the lock.
     * and then we submit the job.
     * If there are no new jobs and we get here, it means the queue
     * for sending jobs is empty, so we might want to ask more jobs.
     * Call the procedure...
     */
    bool new_job = false;    // Indicates whethere there are any new jobs
    Xrsl *xrsl;
    pthread_mutex_lock(new_jobs_lock);

    if (!new_jobs->empty()) {  // See if any new jobs arrived
        notify(INFO) << "New job(s) appeared in queue" << endl;
        // Take the first element in the queue and pop it
        xrsl = new Xrsl(new_jobs->front());
        free(new_jobs->front());
        new_jobs->pop();
        new_job = true;
    }
    pthread_mutex_unlock(new_jobs_lock); // Release lock

    if (new_job) {
        URL *jobid = SubmitJob(xrsl);
        if (!jobid) {
            // Submission failed - do nothing
            // Hey? Are we leaking here sometimes?
            notify(DEBUG) << "Job submission failed - mem leak" << endl;
            return;
        }
        else {
            notify(DEBUG) << jobid->str() << endl;
            // Job was submitted succesfully
            // Create jobstatus object and push onto current_jobs
            JobStatus* js = new JobStatus(xrsl, jobid);
            current_jobs[js->getName()] = js;
            // We need a counter telling how many jobs are
            // in current jobs
            num_current_jobs += 1;
            // Tell instrumentation
            inst->simstart(js->getName());
            inst->tofile();
            notify(INFO) << js->getName() << " is " << jobid->str() << endl;
        }
    } else {
        GetMoreJobs();
    }

    return;
}

void GridJM::GetMoreJobs() {
    if (pipe_write != -1) {
//        notify(INFO) << "Number of jobs in grid: " << num_current_jobs << endl;
        if (num_current_jobs < args->maxjobs) {
            notify(INFO) << "Asking for more jobs from the socket" << endl;
            write(pipe_write, "# getjob 1\n", 11);
        } else {
            notify(INFO) << "Grid is full" << endl;
        }
    }
    return;
}

URL *GridJM::SubmitJob(Xrsl *xrsl) {

    // Check that the xrsl is ok
    try {
        PerformXrslValidation(*xrsl);
    } catch (ARCLibError e) {
        notify(INFO) << e.what() << endl;
        return NULL;
    }

    if (clusters.size() == 0)
        return NULL;

    if (queuelist.size() == 0)
        return NULL;

    /* t_queuelist is made from queuelist by removing current bad
     * clusters from it */
    list<Queue> t_queuelist = queuelist;

    // Remove bad clusters from queues
    cout << "Bad clusters: ";
    for (vector<BCEntry*>::iterator bci = badclusters.begin();
            bci != badclusters.end(); bci++) {
        // Multiple queues in one cluster, remove all.
        // Queues must be removed one at a time
        bool found = true; // At least one round
        while ((*bci)->valid() && found == true) {
            cout << (*bci)->cluster << "(" << (*bci)->valid() << ") ";

            found = false;
            for (list<Queue>::iterator qi = t_queuelist.begin();
                    qi != t_queuelist.end(); qi++) {
                if (strcmp((*qi).cluster.hostname.c_str(),
                                (*bci)->cluster.c_str()) == 0) {
                    t_queuelist.erase(qi);
                    found = true;
                    break;
                }
            }
        }
    }
    cout << "." << endl;

    std::list<Target> targetlist;
    try {
        targetlist = ConstructTargets(t_queuelist, *xrsl);
    } catch (TargetError e) {
        notify(ERROR) << e.what() << endl;
        return NULL;
    }

    PerformStandardBrokering(targetlist);

    JobSubmission submit(*xrsl, targetlist, false);

    URL *jobID;
    string JobName = "unknown";

    if (xrsl->IsRelation("jobname")) {
        JobName = xrsl->GetRelation("jobname").GetSingleValue();
    }

    try {
        jobID = new URL(submit.Submit(20));
        submit.RegisterJobsubmission(queuelist);

    } catch (JobSubmissionError e) {
        notify(WARNING) << "Error: " << e.what() << endl;
        return NULL;
    }
    catch (XrslError e) {
        notify(WARNING) << "Error: " << e.what() << endl;
        return NULL;
    }

    AddJobID(jobID->str(), JobName);

    notify(INFO) << "Submitted job " << *jobID << endl;

    return jobID;
}

void GridJM::UpdateJobStatus() {

    /**
     * Iterate over the jobs, get the status for each one,
     * and act if the state is "FINISHED" or "FAILED"
     */

    /* Get current jobids as a list of strings */
    list<string> jobstrs;
    for (map<string,JobStatus*>::iterator jsi = current_jobs.begin();
            jsi != current_jobs.end(); jsi++)
        jobstrs.push_back((*jsi).second->getJobid()->str());

    bool incremented = false;

    /* Perform query for the list */
    list<Job> joblist = GetJobInfo(jobstrs);

    for (list<Job>::iterator jli = joblist.begin();
        jli != joblist.end(); jli++) {

        string status = "";
        string error = "";

        if (jli->status == "") {
            notify(INFO) << "Job " << jli->id << " is sent but can't be seen (yet)" << endl;
            status = "NOTFOUND";
            error = "";
        }
        else {
            status = jli->status;
            error = jli->errors;
        }

        // Find the job from current_jobs
        JobStatus * current_js = NULL;
        for (map<string,JobStatus*>::iterator jsi = current_jobs.begin();
                jsi != current_jobs.end(); jsi++) {
            if (strcmp((*jsi).second->getJobid()->str().c_str(),
                        jli->id.c_str()) == 0) {
                current_js = (*jsi).second;
                break;
            }
        }

        assert(current_js != NULL); // Supervising unsubmitted jobs

        string prev_status  = current_js->getPrevStatus();
        time_t prev_time    = current_js->getPrevTime();
        time_t current_time = time(NULL);
        current_js->setNewStatus(status);


        notify(INFO) << "Status change " << jli->id << endl;
        notify(INFO) << "  " << prev_status << " => " << status;
        notify(INFO) << " (" << current_time - prev_time << ")" << endl;

        if (strncmp(status.c_str(),"FINISHED", 8) == 0) {

            // Check if error occured during job execution
            if (error.empty()) {
                // Normal job exit
                notify(INFO) << "Job " << jli->id <<
                    " finished correctly and was removed from current job list"
                    << endl;
                // Fetch the job (safe, as not found
                // jobs are not in state FINISHED)
                FetchJob(jli->id);
                // Tell instrumentation
                inst->simstop(current_js->getName(), successfull);
                inst->tofile();
                current_jobs.erase(current_js->getName());
                -- num_current_jobs;
            }
            // Error occured during job execution
            else {
                notify(INFO) << cout <<
                    "Job with id " << jli->id << " failed" << endl;
                // handle job failure here
                bool job_resubmitted = HandleJobFailure(current_js->getName());
            }
        }

        else if ((strncmp(status.c_str(), "INLRMS", 6) == 0) &&
             (strncmp(status.c_str(), "INLRMS:Q", 8) != 0)) {
            // Job is running
        }

        // Check that the job has been doing some progress in
        // grid.
        // The longest status string is MAXSTATUSLENGTH chars
        else if (strncmp(status.c_str(), prev_status.c_str(), MAXSTATUSLENGTH)
                == 0) {
            notify(INFO) << "Job with id " <<
                jli->id << " has not changed" << endl;

            if (current_time - prev_time > args->s_timeout) {
                // Job has been on the same state for
                // too long.
                HandleJobFailure(current_js->getName());
            }
        }

        else if (status.compare("FAILED") == 0) {
            cout << "Job with id " << jli->id << "failed" << endl;
            // handle job failure here
            HandleJobFailure(current_js->getName());
        }
    }
    return;
}

void GridJM::FetchJob(string jobname) {
    pid_t pid = fork();
    if (pid == 0) {
        char* ngget = args->ngget;
        char* dldir = args->dir;
        int rval = execlp(ngget, ngget, "-dir", dldir,
                jobname.c_str(), (char *)NULL);
        perror("execlp()"); // Only return on error
        exit(1);
    }
    // In parent
    // Push the <pid, jobname> -pair to forked_downloads
    PidJob *pj = new PidJob(pid, jobname);
    pthread_mutex_lock(forked_dl_lock);
    forked_downloads->push_back(pj);
    pthread_mutex_unlock(forked_dl_lock);
    return;
}

bool GridJM::HandleJobFailure(string jobname) {

    JobStatus *js = current_jobs[jobname];
    string jobid = js->getJobid()->str();

    cout << "Job " << jobid << " failed - attempting to recover" << endl;

    // Cancel job
    try {
        CancelJob(jobid);
        CleanJob(jobid);
        RemoveJobID(jobid);
    }
    catch (ARCLibError e) {
        notify(ERROR) << "Cancel or clean of job " << jobid << " failed:" << endl;
        notify(ERROR) << e.what();
    }

    // Add a bad cluster to the list of bad clusters
    BCEntry * badc = new BCEntry(js->getJobid()->Host(), BAD_TIMEOUT);
    badclusters.push_back(badc);

    if (js->getAttempts() >= RESUBMIT_ATTEMPTS) {
        // If the job has been submitted more than three times
        // and still failed we throw it out
        notify(DEBUG) << "Job " << jobid << " has been re-submitted "
                 << RESUBMIT_ATTEMPTS << " times - removing it from job list";
        notify(WARNING) << "Job " << jobid <<
            " was removed from current job list, due to maximum number of submissions attempts being reached" << endl;

        // Tell instrumentation
        // state stall + deleted
        inst->simstop(jobname, failed);
        inst->tofile();
        current_jobs.erase(jobname);

        -- num_current_jobs;
        return false;
    }
    // If we get here we attempt to resubmit the job

    Xrsl* xrsl = js->getXrsl();

    URL *new_jobid = SubmitJob(xrsl);
    if (new_jobid != NULL) {
        // Record the resubmission
        inst->evchange(jobname, resubmit);

        js->newSubmission(new_jobid);
        notify(ERROR) << jobid << " was resubmitted as " << new_jobid->str() << endl;

        js->setNewStatus("RESUB");
    }

    else {
        inst->simstop(jobname, submitfail);
        inst->tofile();
        current_jobs.erase(jobname);

        -- num_current_jobs;
        notify(ERROR) << jobid << " resubmission failed";
        return false;
    }

    return true; // jobs was sumbitted again
}

// ========================================================
// BCEntry class from here
// ========================================================
BCEntry::BCEntry(string cl, int to) {
    cluster = cl;
    timeout = to;
    insert_time = time(NULL);
}

bool BCEntry::valid() {
    return time(NULL) - insert_time <= timeout;
}

// ========================================================
// JobStatus class from here
// ========================================================


JobStatus::JobStatus(Xrsl *xrsl, URL *jobid) {

    this->attempts = 0;           // Init number of attempts
    this->xrsl = xrsl;            // Save xrsl
    this->current_jobid = jobid; // Save jobid
    this->prev_status = "INITIAL";
    this->prev_time = time(NULL);

    // Put the cluster name into the cluster vector
    string current_cluster = jobid->Host();
    this->clusters.push_back(current_cluster);

}



JobStatus::~JobStatus() {

    delete xrsl;
    delete current_jobid;
    return;
}


Xrsl* JobStatus::getXrsl() {
    return xrsl;
}


URL *JobStatus::getJobid() {

    return current_jobid;
}


int JobStatus::getAttempts() {

    return attempts;
}

string JobStatus::getPrevStatus() {

    return prev_status;
}

time_t JobStatus::getPrevTime() {

    return prev_time;
}

void JobStatus::setNewStatus(string s) {

    // We only want to update the time if the state has changed
    if (strncmp(prev_status.c_str(), s.c_str(), MAXSTATUSLENGTH) != 0) {
        prev_time = time(NULL);
        prev_status = s;
    }
    return;
}


void JobStatus::newSubmission(URL *new_jobid) {
    clusters.push_back(current_jobid->Host());
    delete current_jobid;
    current_jobid = new_jobid;
    attempts++;
    return;
}

string JobStatus::getName() {
    return xrsl->GetRelation("jobname").GetSingleValue();
}
